<?php
// Helpers generales
session_start();

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;
  $cfg = require __DIR__ . '/config.php';
  $pdo = new PDO($cfg['dsn'], $cfg['user'], $cfg['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

// --- Seguridad / CSRF ---
function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_check(): void {
  $t = $_POST['csrf'] ?? '';
  if (!$t || !hash_equals($_SESSION['csrf'] ?? '', $t)) {
    http_response_code(403);
    exit('CSRF inválido');
  }
}

// --- Flash ---
function flash_set(string $type, string $msg): void {
  $_SESSION['flash'][] = ['type'=>$type,'msg'=>$msg];
}
function flash_get(): array {
  $f = $_SESSION['flash'] ?? [];
  $_SESSION['flash'] = [];
  return $f;
}

// --- Auth ---
function current_user(): ?array { return $_SESSION['user'] ?? null; }
function require_login(): void {
  if (!current_user()) { header('Location: /login.php'); exit; }
  ensure_schema(); // auto-migración ligera
}
function is_admin(): bool { return (bool)(current_user()['is_admin'] ?? false); }

// --- URLs ---
function url(string $path): string {
  if (preg_match('#^https?://#i', $path)) return $path;
  return '/' . ltrim($path, '/');
}
function redirect(string $path): void {
  header('Location: ' . url($path));
  exit;
}
function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// --- MIGRACIONES LIGERAS (sin phpMyAdmin) ---
function table_has_column(string $table, string $col): bool {
  $pdo = db();
  $st = $pdo->prepare("SELECT COUNT(*) c FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME=? AND COLUMN_NAME=?");
  $st->execute([$table,$col]);
  return ((int)$st->fetch()['c']) > 0;
}

function ensure_schema(): void {
  // ejecuta una vez por sesión para no cargar
  if (!empty($_SESSION['schema_ok_v1'])) return;
  $pdo = db();

  // users.avatar_url (para DiceBear o subida)
  if (table_exists('users') && !table_has_column('users','avatar_url')) {
    $pdo->exec("ALTER TABLE users ADD COLUMN avatar_url VARCHAR(255) NULL");
  }

  // company_settings.logo_path
  if (table_exists('company_settings') && !table_has_column('company_settings','logo_path')) {
    $pdo->exec("ALTER TABLE company_settings ADD COLUMN logo_path VARCHAR(255) NULL");
  }

  // clients: código postal / localidad / provincia
  if (table_exists('clients')) {
    if (!table_has_column('clients','postal_code')) $pdo->exec("ALTER TABLE clients ADD COLUMN postal_code VARCHAR(10) NULL");
    if (!table_has_column('clients','localidad'))   $pdo->exec("ALTER TABLE clients ADD COLUMN localidad VARCHAR(120) NULL");
    if (!table_has_column('clients','provincia'))   $pdo->exec("ALTER TABLE clients ADD COLUMN provincia VARCHAR(120) NULL");
  }

  $_SESSION['schema_ok_v1'] = 1;
}

function table_exists(string $table): bool {
  $pdo=db();
  $st=$pdo->prepare("SELECT COUNT(*) c FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA=DATABASE() AND TABLE_NAME=?");
  $st->execute([$table]);
  return ((int)$st->fetch()['c']) > 0;
}

// Avatar mostrado en UI
function avatar_url(array $user): string {
  $u = trim((string)($user['avatar_url'] ?? ''));
  if ($u !== '') return $u;
  // fallback: DiceBear por username si no tiene avatar guardado
  $seed = rawurlencode((string)($user['username'] ?? 'usuario'));
  return "https://api.dicebear.com/7.x/notionists/svg?seed={$seed}";
}
